% WT2_model()

% This is the main function of the WT-2 model
% Two types of molecular clutches: talin-based, non-talin-based
% k_off is different for both types of molecular clutches 
% This program runs Runge-Kutta calculations
% Elastic characteristics of clutches are modelled based on single-molecule
% data measured on talin


close all

view_angle = [80, 68];

%%%% Model parameters %%%%

n_steps = 100;                           % number of points to calculate, (n_steps) x (n_steps)
index = 0 : (n_steps - 1);
v_actin_step = log(300) / (n_steps - 1);
v_actin = 1 * exp(index * v_actin_step); % [nm/s], retrograde actin flow


%%% Clutch parameters %%%
 
k_on_1 = 0.0044;         % [1/s], formation rate of non-talin-based molecular clutches
k_on_2 = 0.091; %0.091;  % [1/s], formation rate of talin-based molecular clutches

% [Artola 2016] data
k_off_1_non_talin = 0.3755; %0.3755;          % [1/s], dissociation rate of talin-based molecular clutches (catch bond)
k_off_2_non_talin = 5.433e-05; %5.433e-05;    % [1/s], dissociation rate of talin-based molecular clutches (catch bond)
dissociation_force_1_non_talin = -38.09;      % [pN], characteristic dissociation force of talin-based molecular clutches (catch bond)
dissociation_force_2_non_talin = 4.411;       % [pN], characteristic dissociation force of talin-based molecular clutches (catch bond)

k_off_1_talin = 0.3755; %0.3755;                              % [1/s], dissociation rate of non-talin-based molecular clutches (catch bond)
k_off_2_talin = 5.433e-05; %5.433e-05;                        % [1/s], dissociation rate of non-talin-based molecular clutches (catch bond)
dissociation_force_1_talin = dissociation_force_1_non_talin;  % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)
dissociation_force_2_talin = dissociation_force_2_non_talin;  % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)

myosin_unloaded_rate = 120; % [nm/s], = 110 nm/s [Artola, Nature Materials, 2014], = 110 nm/s [Artola, Nat Cell Biol, 2016], ~200 nm/s [Beausang et al, Biophys. J., 2008, 95:5820]
myosin_stall_force = 2;     % [pN], = 2 pN [Artola, Nature Materials, 2014], = 2 pN [Artola, Nat Cell Biol, 2016]
myosin_density = 88;        % Myosin density, [1/um^2]. Based on Alieva and Hu papers.

clutches_density = 300; %300;   % [1/um^2], density of sites available for molecular clutch formation (clutches max density = integrin density)

clutches_stiffness_1 = 0.9;     % [pN/nm], spring constant of non-talin-based molecular clutches
l_yield_1 = 160;                % [nm], l_y value of the nonlinear behaviour of non-talin-based molecular clutches

a_1_1 = 33;       % [nm], characteristic radius of single non-talin based clutch adhesion site
a_1_2 = 0.25;     % [nm], characteristic radius of single talin based clutch adhesion site
 
a_2 = 1700;  % [nm], characteristic size of cell adhesion complexes, = 550 nm [Artola, Nature Materials, 2014], = 1700 nm [Artola, Nat Cell Biol, 2016]

Young_modulus_max = 10^10; %1e+06;
Young_modulus_min = 10^2; %1e+02;
% substrate_stiffness_max = Young_modulus_max * ( 4 * pi / 9 * a_1_1 * 10^(-9) * 1000 ); % [Pa]
substrate_stiffness_min_1 = Young_modulus_min * ( 4 * pi / 9 * a_1_1 * 10^(-9) * 1000 ); % [Pa]
substrate_stiffness_min_2 = Young_modulus_min * ( 4 * pi / 9 * a_1_2 * 10^(-9) * 1000 ); % [Pa]
substrate_stiffness_step = log(Young_modulus_max / Young_modulus_min) / (n_steps - 1);
substrate_stiffness_1 = substrate_stiffness_min_1 * exp(index * substrate_stiffness_step)';    % [pN/nm]
substrate_stiffness_2 = substrate_stiffness_min_2 * exp(index * substrate_stiffness_step)';    % [pN/nm]
Young_modulus = substrate_stiffness_1 / ( 4 * pi / 9 * a_1_1 * 10^(-9) * 1000 ); % [Pa]
effective_stiffness_1 = substrate_stiffness_1 * clutches_stiffness_1 ./ (substrate_stiffness_1 + clutches_stiffness_1);
 
myosin_number   = pi * a_2^2 / 1e+06 * myosin_density;    % total number of myosin II motors
clutches_number = pi * a_2^2 / 1e+06 * clutches_density;  % total number of sites available for molecular clutch formation
 
myosin_force = myosin_stall_force * myosin_number * (1 - v_actin / myosin_unloaded_rate);  % force-velocity curve of myosin II
myosin_force(myosin_force < -0.5) = -0.5;

s1 = length(Young_modulus);
myosin_force = repmat(myosin_force, s1, 1);
Cell_traction_myosin = myosin_force * 1e-12 / pi / (a_2 * 1e-09)^2; % [Pa]
 
 
%%% Clutch transition model parameters, single mechano-sensitive domain %%%
 
N_unfolding_domains = 1;  % Here only the mechanosensitive R3 domain of talin is assumed to unfold under mechanical load
 
N_states = 2^N_unfolding_domains;                              % domain folded and domain unfolded
list_of_clutch_states = ones(N_states, N_unfolding_domains);
for i = 1 : N_states
    list_of_clutch_states(i, :) = bitget(i, 1:N_unfolding_domains);  % state = 0 - folded, 1 - unfolded
end
 
k_0_unfold = 0.018;      % [s^-1], zero force unfolding rate of talin [Yao Mingxi article]
k_0_refold = 22.2;       % [s^-1], zero force refolding rate of talin [Yao Mingxi article]
unfolding_force = 0.7;   % [pN], characteristic unfolding force of talin [Yao Mingxi article]
 
if( size(k_0_unfold, 2) ~= N_unfolding_domains || size(k_0_refold, 2) ~= N_unfolding_domains || size(unfolding_force, 2) ~= N_unfolding_domains) 
    disp('Error, wrong length of k_0_unfold, k_0_refold or unfolding_force arrays')
end
 
L_p = 1;                % [nm] persistence length of a peptide chain, [Winardhi et al, Biophys. J., 2016, 111:2349].
linker_size = 37;       % 37 nm - all domains folded, 217 nm - 3 vinculin binding domains unfolded (VD1 vinculin experiment) the contour length of the flexible talin neck domain in [nm], contour length of talin with folded domains is 97nm [Liu et al, Tony group, PNAS, 2015, E4864]
domain_contour_length = 49.6; % [nm], the contour lengths of the unfolded R3 domain of talin
 
talin_globular_domain_number = 12;   % 12 - all domains folded, 10 - 3 vinculin binding domains unfolded (VD1 vinculin experiment)
integrin_globular_domain_number = 0; % does not play an important role
globular_domain_size = 5;            % mean size of folded domains in [nm]
 
kT = 1.38e-23 * 310 / 1e-21;                            % [pN*nm], thermal energy
clutch_delta_G_0 = kT * log(k_0_refold ./ k_0_unfold ); % delta_G_0 = G_unfolded - G_folded > 0 in [pN*nm] units
 
number_added_sites_per_unfolded_clutch = 200; %200;

% talin_K0_unfold = [4.2E-6, 1.7E-8, 0.018, 4.2E-6, 2.5E-5, 2.5E-5, 4.2E-06, 4.2E-6, 2.5E-5, 2.5E-5, 1.7E-8];  % in units of  s^-1, zero force unfolding rates from Table 1
% talin_K0_refold = [0.11, 0.019, 22.2, 0.46, 1.0, 1.0, 0.39, 0.93, 0.93, 0.93, 0.93 ];                        % in units of s^-1, zero force refolding rates from Table 1
% talin_X_unfold  = [3.1, 3.40, 5.7, 3.1, 4.1, 4.1, 3.1, 3.1, 4.1, 4.1, 3.40];                                 % in units of nm, the unfolding transition distance from Table 1
% Rod_contour_length = [69.2, 52.4, 49.6, 52.4, 64, 60.4, 91.6, 66.8, 63.2, 66.4, 62.8]; % [nm]


%%%%%%% Experimental data from [Artola, Nat Cell Biol, 2016] %%%%%%%

figure(100)

data_x = [620
2400
4700
12000
15000
29000];

Cell_traction_KD_0_blebbistatin = [
20
54.8
83
47
33
37];

Cell_traction_WT_0_blebbistatin = [
22.4
53.2
92
89
88
137];

Cell_traction_WT_15_blebbistatin = [
21.3
50.3
60.5
75.4
72
60];

Cell_traction_WT_50_blebbistatin = [
19
29
38
45
40
48];

Cell_traction_KD_1_fibronectin = [21.1
51.4
39.8
34.8
30.1
36.8];

Cell_traction_KD_100_fibronectin = [19.5
47.7
81.1
116.6
60.7
37.7];

Cell_traction_WT_1_fibronectin = [
19.7
55.6
60.9
78.6
88.8
101.5];

Cell_traction_WT_100_fibronectin = [
21.5
53.3
98.5
134.7
112.8
143.1];

Cell_traction_WT_10_fibronectin_VD1_vinculin = [
22.5
49.4
86.5
64
45.3
48.6];

semilogx(data_x, Cell_traction_KD_0_blebbistatin, 'bo')
hold on
semilogx(data_x, Cell_traction_WT_0_blebbistatin, 'ro')
% semilogx(data_x, Cell_traction_WT_15_blebbistatin, 'go')
% semilogx(data_x, Cell_traction_WT_50_blebbistatin, 'co')
% semilogx(data_x, Cell_traction_WT_1_fibronectin, 'yo')
% semilogx(data_x, Cell_traction_WT_100_fibronectin, 'mo')
% semilogx(data_x, Cell_traction_WT_10_fibronectin_VD1_vinculin, 'ko')
% hold on
% semilogx(data_x, Cell_traction_KD_1_fibronectin, 'go')
% semilogx(data_x, Cell_traction_KD_100_fibronectin, 'co')
title('Cell traction')
xlabel('Young modulus, Pa') 
ylabel('Cell traction, Pa')

figure(101)

data_x = [
2400
4400
12000
15000
29000];

Cell_traction_KD_0_blebbistatin = [
71
45
68
84
98];

Cell_traction_WT_0_blebbistatin = [
76
48
56
54
25];

    Young_flow = [
    260
    730
    1090
    1280
    24000
    57000];

    Flow = [
    77.6419
    75.8952
    84.9782
    101.9214
    110.4803
    105.2402];

semilogx(data_x, Cell_traction_KD_0_blebbistatin, 'bo')
hold on
semilogx(data_x, Cell_traction_WT_0_blebbistatin, 'ro')
hold on
% semilogx(Young_flow, Flow, 'go')
title('Actin flow rate')
xlabel('Young modulus, Pa') 
ylabel('Actin flow rate, nm/s')


%%%%%%%% Runge-Kutta calculations %%%%%%%%

break_flag = false;

force_catch_full_model = zeros(n_steps, n_steps);

k_on_f = k_on_2 * k_0_refold / ( k_0_refold + k_0_unfold );
k_on_u = k_on_2 * k_0_unfold / ( k_0_refold + k_0_unfold );
k_on_vector = [k_on_u; k_on_f; k_on_1];  % 1st row of the vector = unfolded talin, 2nd = folded talin, 3rd = KD clutches
k_on = sum(k_on_vector, 'all');

%%% Initial p_on vector (up to a 'C' multiplier) %%%
p_0_on = k_on_vector / k_on; % 1st matrix row = unfolded talin, 2nd matrix row = folded talin, 3rd matrix row = KD clutches
        
for i = 1 : n_steps
    disp(['E = ' num2str(Young_modulus(i)) ' Pa'])
    
    for j = 1 : n_steps
        v = v_actin(j);
        disp(['   v = ' num2str(v) ' nm/s'])
        
        clutch_extension = 0 : v/200 : min([1000, 150 + sqrt(10000 / substrate_stiffness_2(i)), 20*v]); % row, [nm]
%         clutch_extension = 0 : v/200 : min([1000, 20*v]); % row, [nm]
        [tabulated_clutch_forces, tabulated_clutch_free_energies] = tabulate_clutch_force_and_energy(substrate_stiffness_2(i), N_unfolding_domains, ...
            clutch_extension, linker_size, domain_contour_length, clutch_delta_G_0, list_of_clutch_states, kT, L_p, talin_globular_domain_number, ...
            integrin_globular_domain_number, globular_domain_size); 
        % tabulated_clutch_forces        - in [pN], each row corresponds to a clutch state, different extension
        % tabulated_clutch_free_energies - in [pN*nm], each row corresponds to a clutch state, different extension

        force_1 = force_with_yield_point(clutch_extension, l_yield_1, effective_stiffness_1(i));
        
        RK_N_steps = floor( ( size(clutch_extension, 2) - 1 ) / 2 );
        RK_half_step = clutch_extension(2) - clutch_extension(1);
        p_on_all  = zeros( N_states + 1, RK_N_steps + 1 );
%         k_off_all = zeros( N_states + 1, RK_N_steps );
        p_on_all(:, 1) = p_0_on;
        
        %%% Runge-Kutta %%%
        
        for k = 1 : RK_N_steps
            %%% Build transition matrix %%%
            
            A_step = zeros(3, 3);
            A_half_step = zeros(3, 3);
            A_zero_step = zeros(3, 3);
            
            force_step      = tabulated_clutch_forces(:, 2*k + 1); 
            force_half_step = tabulated_clutch_forces(:, 2*k); 
            force_zero_step = tabulated_clutch_forces(:, 2*k - 1); 
            
            free_energy_step      = tabulated_clutch_free_energies(:, 2*k + 1); 
            free_energy_half_step = tabulated_clutch_free_energies(:, 2*k); 
            free_energy_zero_step = tabulated_clutch_free_energies(:, 2*k - 1); 
            
            k_off_talin_step      = k_off_1_talin * exp( force_step / dissociation_force_1_talin)      + k_off_2_talin * exp( force_step / dissociation_force_2_talin);
            k_off_talin_half_step = k_off_1_talin * exp( force_half_step / dissociation_force_1_talin) + k_off_2_talin * exp( force_half_step / dissociation_force_2_talin);
            k_off_talin_zero_step = k_off_1_talin * exp( force_zero_step / dissociation_force_1_talin) + k_off_2_talin * exp( force_zero_step / dissociation_force_2_talin);
            
            k_off_non_talin_step      = k_off_1_non_talin * exp( force_1(2*k + 1) / dissociation_force_1_non_talin) + k_off_2_non_talin * exp( force_1(2*k + 1) / dissociation_force_2_non_talin);
            k_off_non_talin_half_step = k_off_1_non_talin * exp( force_1(2*k) / dissociation_force_1_non_talin)     + k_off_2_non_talin * exp( force_1(2*k) / dissociation_force_2_non_talin);
            k_off_non_talin_zero_step = k_off_1_non_talin * exp( force_1(2*k - 1) / dissociation_force_1_non_talin) + k_off_2_non_talin * exp( force_1(2*k - 1) / dissociation_force_2_non_talin);
            
            k_off_talin_step( k_off_talin_step > 1e+2 )           = 1e+2;
            k_off_talin_half_step( k_off_talin_half_step > 1e+2 ) = 1e+2;
            k_off_talin_zero_step( k_off_talin_zero_step > 1e+2 ) = 1e+2;
            
            k_off_non_talin_step( k_off_non_talin_step > 1e+2 )           = 1e+2;
            k_off_non_talin_half_step( k_off_non_talin_half_step > 1e+2 ) = 1e+2;
            k_off_non_talin_zero_step( k_off_non_talin_zero_step > 1e+2 ) = 1e+2;
            
%             k_off_all(1:2, k) = k_off_talin_zero_step;
%             k_off_all(3, k)   = k_off_non_talin_zero_step;
            
            k_unfold_step      = k_0_unfold * exp( force_step(2) / unfolding_force);
            k_unfold_half_step = k_0_unfold * exp( force_half_step(2) / unfolding_force);
            k_unfold_zero_step = k_0_unfold * exp( force_zero_step(2) / unfolding_force);

            k_unfold_step( k_unfold_step > 1e+2 )           = 1e+2;
            k_unfold_half_step( k_unfold_half_step > 1e+2 ) = 1e+2;
            k_unfold_zero_step( k_unfold_zero_step > 1e+2 ) = 1e+2;
            
            k_refold_step      = k_unfold_step * exp( (free_energy_step(1) - free_energy_step(2)) / kT );
            k_refold_half_step = k_unfold_half_step * exp( (free_energy_half_step(1) - free_energy_half_step(2)) / kT );
            k_refold_zero_step = k_unfold_zero_step * exp( (free_energy_zero_step(1) - free_energy_zero_step(2)) / kT );
 
            A_step(1, 1) = -k_off_talin_step(1) - k_refold_step;  % 1st matrix row = unfolded talin state
            A_step(2, 2) = -k_off_talin_step(2) - k_unfold_step;  % 2nd matrix row = folded talin state
            A_step(1, 2) = k_unfold_step;
            A_step(2, 1) = k_refold_step;
            A_step(3, 3) = -k_off_non_talin_step;
            A_step = A_step / v;
      
            A_half_step(1, 1) = -k_off_talin_half_step(1) - k_refold_half_step;
            A_half_step(2, 2) = -k_off_talin_half_step(2) - k_unfold_half_step;
            A_half_step(1, 2) = k_unfold_half_step;
            A_half_step(2, 1) = k_refold_half_step;
            A_half_step(3, 3) = -k_off_non_talin_half_step;
            A_half_step = A_half_step / v;
            
            A_zero_step(1, 1) = -k_off_talin_zero_step(1) - k_refold_zero_step;
            A_zero_step(2, 2) = -k_off_talin_zero_step(2) - k_unfold_zero_step;
            A_zero_step(1, 2) = k_unfold_zero_step;
            A_zero_step(2, 1) = k_refold_zero_step;
            A_zero_step(3, 3) = -k_off_non_talin_zero_step;
            A_zero_step = A_zero_step / v;

            k_1 = A_zero_step * p_on_all(:, k);
            k_2 = A_half_step * ( p_on_all(:, k) + RK_half_step * k_1 );
            k_3 = A_half_step * ( p_on_all(:, k) + RK_half_step * k_2 );
            k_4 = A_step * ( p_on_all(:, k) + 2 * RK_half_step * k_3 );
            
            p_on_all(:, k + 1) = p_on_all(:, k) + RK_half_step / 3 * ( k_1 + 2 * k_2 + 2 * k_3 + k_4 );
           
            if(sum(p_on_all(:, k+1)) < 1e-06) 
                break
            end
        end
        
        %%% Finding C constant %%%
        
        I = k_on * 2 * RK_half_step * sum( p_on_all(:, 1 : RK_N_steps), 'all' );
        C = k_on / ( v + I );
        p_on_all = C * p_on_all;
        
        %%% Calculating the average traction force %%%
        
        I_C = 2 * RK_half_step * sum( p_on_all(1, 1 : RK_N_steps), 'all' );
        denominator = 1 - number_added_sites_per_unfolded_clutch * I_C;
        if(denominator < 0.01)
            denominator = 0.01;
        end
        clutches_number_total = clutches_number / denominator;
        force_catch_full_model(i, j) = clutches_number_total * 2 * RK_half_step * sum( tabulated_clutch_forces(:, 1 : 2 : 2*RK_N_steps-1 ) .* p_on_all( 1 : 2, 1 : RK_N_steps), 'all' );
        force_catch_full_model(i, j) = force_catch_full_model(i, j) + clutches_number_total * 2 * RK_half_step * sum( force_1(1 : 2 : 2*RK_N_steps-1 ) .* p_on_all( 3, 1 : RK_N_steps), 'all' );
    end
end

Cell_traction_substrate = force_catch_full_model * 1e-12 / pi / (a_2 * 1e-09)^2; % [Pa]
    
figure(22)
    
force_catch_full_model_graph = force_catch_full_model;
force_catch_full_model_graph(force_catch_full_model_graph > 2001) = 2001;
surf(log10(v_actin), log10(Young_modulus), force_catch_full_model_graph)
colormap jet
shading interp
freezeColors()
hold on
surf(log10(v_actin), log10(Young_modulus), myosin_force, 'FaceColor', [0.7 0.7 0.7], 'EdgeColor', 'none');
%     view(view_angle)
view(58, 57)
axis([log10(v_actin(1)) log10(v_actin(end)) log10(Young_modulus_min) log10(Young_modulus_max) 0 min([2000 1.3 * max(force_catch_full_model, [], 'all')])])
%     axis([log10(v_actin(1)) log10(v_actin(end)) 2 6 0 1000])


figure(5) 
 
M = contour(log10(v_actin), log10(Young_modulus), force_catch_full_model-myosin_force, [0 0], 'k');
      
N_max = length(M);
s = 1;
while s <= N_max
    x_zero_contour = M(1, (s + 1) : ( s + M(2,s) ) );
    y_zero_contour = M(2, (s + 1) : ( s + M(2,s) ) );
    s = s + M(2,s) + 1;
        
    Young_modulus_temp = 10.^y_zero_contour;
        
    velocity_zero_contour = 10.^x_zero_contour;
    z_zero_contour = (1 - velocity_zero_contour / myosin_unloaded_rate) * myosin_stall_force * myosin_number;
    z_zero_contour(z_zero_contour < 0) = 0;
    
    Young_modulus_force_catch_bond_model_zero_contour = Young_modulus_temp;
    force_catch_bond_model_zero_contour = z_zero_contour;

    index_old = 1;
    flag = 1;
    figure(22)
    for j = 2 : length(Young_modulus_temp) - 1
        if( Young_modulus_temp(j) < Young_modulus_temp(j-1) && Young_modulus_temp(j) < Young_modulus_temp(j+1))
            index_new = j;
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k', 'LineWidth', 2)
            plot3(x_zero_contour(index_new), y_zero_contour(index_new), z_zero_contour(index_new), 'wo', 'MarkerFaceColor','white', 'MarkerEdgeColor', 'black', 'MarkerSize', 5)
            index_old = index_new;
            flag = 0;
        end
        if( Young_modulus_temp(j) > Young_modulus_temp(j-1) && Young_modulus_temp(j) > Young_modulus_temp(j+1))
            index_new = j;
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'w', 'LineWidth', 2)
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k--', 'LineWidth', 2)
            plot3(x_zero_contour(index_new), y_zero_contour(index_new), z_zero_contour(index_new), 'wo', 'MarkerFaceColor','white', 'MarkerEdgeColor', 'black', 'MarkerSize', 5)
            index_old = index_new;
            flag = 1;
        end
    end
        
    index_new = length(Young_modulus_temp);
    if(flag == 1)
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k', 'LineWidth', 2)
    else
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'w', 'LineWidth', 2)
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k--', 'LineWidth', 2)
    end
        
    title('Velocity-stiffness-force, catch bond model')
    xlabel('Lg movement rate, nm/s') 
    ylabel('Lg(E)') 
    zlabel('Force, pN') 
%         view(view_angle)
    view(58, 57)
        
    if(break_flag)
        break
    end
end
    
  
figure(24)
    
Cell_traction_substrate_graph = Cell_traction_substrate;
Cell_traction_substrate_graph(Cell_traction_substrate_graph > 301) = 301;
surf(log10(v_actin), log10(Young_modulus), Cell_traction_substrate_graph)
colormap jet
shading interp
freezeColors()
hold on
surf(log10(v_actin), log10(Young_modulus), Cell_traction_myosin, 'FaceColor', [0.7 0.7 0.7], 'EdgeColor', 'none');
%     axis([log10(v_actin(1)) log10(v_actin(end)) log10(Young_modulus(1))  log10(Young_modulus(end)) 0 1.1 * max(Cell_traction_substrate, [], 'all')])
axis([log10(v_actin(1)) log10(v_actin(end)) log10(Young_modulus_min) log10(Young_modulus_max) 0 min([300, 1.3 * max(Cell_traction_substrate_graph, [], 'all')])])
%     axis([log10(v_actin(1)) log10(200) 2  6 0 400])
    % colormap gray
    % shading interp

title('Velcotiy-stiffness-force, full catch model')
xlabel('Lg movement rate, nm/s') 
ylabel('Lg(E), Pa') 
zlabel('Cell traction, Pa') 
view(view_angle)

figure(5)

surf(log10(v_actin), log10(Young_modulus), Cell_traction_substrate-Cell_traction_myosin)
colormap jet
shading interp
hold on
    
M = contour(log10(v_actin), log10(Young_modulus), Cell_traction_substrate-Cell_traction_myosin, [0 0], 'k');
    
N_max = length(M);
s = 1;
while s <= N_max
    x_zero_contour = M(1, (s + 1) : ( s + M(2,s) ) );
    y_zero_contour = M(2, (s + 1) : ( s + M(2,s) ) );
    s = s + M(2,s) + 1;
       
    Young_modulus_temp = 10.^y_zero_contour;
                
    velocity_zero_contour = 10.^x_zero_contour;
    force_zero_contour = (1 - velocity_zero_contour / myosin_unloaded_rate) * myosin_stall_force * myosin_number;
        
    z_zero_contour = force_zero_contour * 1e-12 / pi / (a_2 * 1e-09)^2; % [Pa]
    z_zero_contour(z_zero_contour < 0) = 0;
    
    index_old = 1;
    flag = 1;
        
    figure(24)
    for j = 2 : length(Young_modulus_temp) - 1
        if( Young_modulus_temp(j) < Young_modulus_temp(j-1) && Young_modulus_temp(j) < Young_modulus_temp(j+1))
            index_new = j;
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k', 'LineWidth', 2)
            plot3(x_zero_contour(index_new), y_zero_contour(index_new), z_zero_contour(index_new), 'wo', 'MarkerFaceColor','white', 'MarkerEdgeColor', 'black', 'MarkerSize', 5)
            index_old = index_new;
            flag = 0;
        end
        if( Young_modulus_temp(j) > Young_modulus_temp(j-1) && Young_modulus_temp(j) > Young_modulus_temp(j+1))
            index_new = j;
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'w', 'LineWidth', 2)
            plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k--', 'LineWidth', 2)
            plot3(x_zero_contour(index_new), y_zero_contour(index_new), z_zero_contour(index_new), 'wo', 'MarkerFaceColor','white', 'MarkerEdgeColor', 'black', 'MarkerSize', 5)
            index_old = index_new;
            flag = 1;
        end
    end
        
    index_new = length(Young_modulus_temp);
    if(flag == 1)
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k', 'LineWidth', 2)
    else
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'w', 'LineWidth', 2)
        plot3(x_zero_contour(index_old : index_new), y_zero_contour(index_old : index_new), z_zero_contour(index_old : index_new), 'k--', 'LineWidth', 2)
    end
        
    title('Velocity-stiffness-traction, catch bond model')
    xlabel('Lg movement rate, nm/s') 
    ylabel('Lg(E)') 
    zlabel('Cell traction, Pa') 
    view(view_angle)
                
    Actin_flow_zero_contour = 10.^x_zero_contour;
    Young_modulus_zero_contour = 10.^y_zero_contour;
    Cell_traction_zero_contour = z_zero_contour;
    
    substrate_stiffness_zero_contour_1 = ( 4 * pi / 9 * a_1_1 * 10^(-9) * 1000 ) * Young_modulus_zero_contour; % [pN/nm]
    substrate_stiffness_zero_contour_2 = ( 4 * pi / 9 * a_1_2 * 10^(-9) * 1000 ) * Young_modulus_zero_contour;
    
    effective_stiffness_zero_contour_1 = substrate_stiffness_zero_contour_1 * clutches_stiffness_1 ./ (substrate_stiffness_zero_contour_1 + clutches_stiffness_1);
    
    
    figure(6)
        
    F_average_zero_contour                   = zeros( N_states + 1, length(Young_modulus_zero_contour) );
    Loading_rate_average_zero_contour        = zeros( N_states + 1, length(Young_modulus_zero_contour) );
    Loading_rate_average_square_zero_contour = zeros( N_states + 1, length(Young_modulus_zero_contour) );
    Loading_rate_std_zero_contour            = zeros( N_states + 1, length(Young_modulus_zero_contour) );
    Clutches_on_zero_contour                 = zeros( N_states + 1, length(Young_modulus_zero_contour) );
    
    parfor j = 1 : length(Young_modulus_zero_contour)
        v = Actin_flow_zero_contour(j);

        clutch_extension = 0 : v/200 : min([1000, 150 + sqrt(10000 / substrate_stiffness_zero_contour_2(j)), 20*v]);
%         clutch_extension = 0 : v/200 : min([1000, 20*v]);% row, [nm]
        [tabulated_clutch_forces, tabulated_clutch_free_energies] = tabulate_clutch_force_and_energy(substrate_stiffness_zero_contour_2(j), N_unfolding_domains, ...
            clutch_extension, linker_size, domain_contour_length, clutch_delta_G_0, list_of_clutch_states, kT, L_p, talin_globular_domain_number, ...
            integrin_globular_domain_number, globular_domain_size); 
        % tabulated_clutch_forces        - in [pN], each row corresponds to a clutch state, different extension
        % tabulated_clutch_free_energies - in [pN*nm], each row corresponds to a clutch state, different extension
        
        force_1 = force_with_yield_point(clutch_extension, l_yield_1, effective_stiffness_zero_contour_1(j));
        tabulated_clutch_forces_expanded = [tabulated_clutch_forces; force_1];
        
        tabulated_force_derivative = ( circshift(tabulated_clutch_forces_expanded, -1, 2) - circshift(tabulated_clutch_forces_expanded, 1, 2) ) / 2 / clutch_extension(2);
        tabulated_force_derivative(:, 1)   = ( tabulated_clutch_forces_expanded(:, 2) - tabulated_clutch_forces_expanded(:, 1) )       / clutch_extension(2);
        tabulated_force_derivative(:, end) = ( tabulated_clutch_forces_expanded(:, end) - tabulated_clutch_forces_expanded(:, end-1) ) / clutch_extension(2);
        % tabulated_force_derivative        - in [pN/nm], each row corresponds to a clutch state, different extension
        
%         disp([ 'tabulated_clutch_forces_expanded_size = ' num2str( size(tabulated_clutch_forces_expanded) ) ])
%         disp([ 'tabulated_force_derivative_size = ' num2str( size(tabulated_force_derivative) ) ])

        %%% Initial p_on vector (up to a 'C' multiplier) %%%

        RK_N_steps = floor( ( size(clutch_extension, 2) - 1 ) / 2 );
        RK_half_step = clutch_extension(2) - clutch_extension(1);
        p_on_all  = zeros( N_states + 1, RK_N_steps + 1 );
%         k_off_all = zeros( N_states + 1, RK_N_steps );
        p_on_all(:, 1) = p_0_on;
        
%         disp(['j = ' num2str(j)])

        %%% Runge-Kutta %%%
        
        for k = 1 : RK_N_steps
            %%% Build transition matrix %%%

            A_step = zeros(3, 3);
            A_half_step = zeros(3, 3);
            A_zero_step = zeros(3, 3);
            
%             force_step      = tabulated_clutch_forces_expanded(:, 2*k + 1); 
%             force_half_step = tabulated_clutch_forces_expanded(:, 2*k); 
%             force_zero_step = tabulated_clutch_forces_expanded(:, 2*k - 1); 

            force_step      = tabulated_clutch_forces(:, 2*k + 1); 
            force_half_step = tabulated_clutch_forces(:, 2*k); 
            force_zero_step = tabulated_clutch_forces(:, 2*k - 1); 

            free_energy_step      = tabulated_clutch_free_energies(:, 2*k + 1); 
            free_energy_half_step = tabulated_clutch_free_energies(:, 2*k); 
            free_energy_zero_step = tabulated_clutch_free_energies(:, 2*k - 1); 
            
            k_off_talin_step      = k_off_1_talin * exp( force_step / dissociation_force_1_talin)      + k_off_2_talin * exp( force_step / dissociation_force_2_talin);
            k_off_talin_half_step = k_off_1_talin * exp( force_half_step / dissociation_force_1_talin) + k_off_2_talin * exp( force_half_step / dissociation_force_2_talin);
            k_off_talin_zero_step = k_off_1_talin * exp( force_zero_step / dissociation_force_1_talin) + k_off_2_talin * exp( force_zero_step / dissociation_force_2_talin);
            
            k_off_non_talin_step      = k_off_1_non_talin * exp( force_1(2*k + 1) / dissociation_force_1_non_talin) + k_off_2_non_talin * exp( force_1(2*k + 1) / dissociation_force_2_non_talin);
            k_off_non_talin_half_step = k_off_1_non_talin * exp( force_1(2*k) / dissociation_force_1_non_talin)     + k_off_2_non_talin * exp( force_1(2*k) / dissociation_force_2_non_talin);
            k_off_non_talin_zero_step = k_off_1_non_talin * exp( force_1(2*k - 1) / dissociation_force_1_non_talin) + k_off_2_non_talin * exp( force_1(2*k - 1) / dissociation_force_2_non_talin);
            
            k_off_talin_step( k_off_talin_step > 1e+2 )           = 1e+2;
            k_off_talin_half_step( k_off_talin_half_step > 1e+2 ) = 1e+2;
            k_off_talin_zero_step( k_off_talin_zero_step > 1e+2 ) = 1e+2;
            
            k_off_non_talin_step( k_off_non_talin_step > 1e+2 )           = 1e+2;
            k_off_non_talin_half_step( k_off_non_talin_half_step > 1e+2 ) = 1e+2;
            k_off_non_talin_zero_step( k_off_non_talin_zero_step > 1e+2 ) = 1e+2;
            
%             k_off_all(1:2, k) = k_off_zero_step;
%             k_off_all(3, k)   = k_off_1_zero_step;

            k_unfold_step      = k_0_unfold * exp( force_step(2) / unfolding_force);
            k_unfold_half_step = k_0_unfold * exp( force_half_step(2) / unfolding_force);
            k_unfold_zero_step = k_0_unfold * exp( force_zero_step(2) / unfolding_force);

            k_unfold_step( k_unfold_step > 1e+2 ) = 1e+2;
            k_unfold_half_step( k_unfold_half_step > 1e+2 ) = 1e+2;
            k_unfold_zero_step( k_unfold_zero_step > 1e+2 ) = 1e+2;
            
            k_refold_step      = k_unfold_step * exp( (free_energy_step(1) - free_energy_step(2)) / kT );
            k_refold_half_step = k_unfold_half_step * exp( (free_energy_half_step(1) - free_energy_half_step(2)) / kT );
            k_refold_zero_step = k_unfold_zero_step * exp( (free_energy_zero_step(1) - free_energy_zero_step(2)) / kT );
 
            A_step(1, 1) = -k_off_talin_step(1) - k_refold_step;  % 1st matrix row = unfolded talin state
            A_step(2, 2) = -k_off_talin_step(2) - k_unfold_step;  % 2nd matrix row = folded talin state
            A_step(1, 2) = k_unfold_step;
            A_step(2, 1) = k_refold_step;
            A_step(3, 3) = -k_off_non_talin_step;
            A_step = A_step / v;
      
            A_half_step(1, 1) = -k_off_talin_half_step(1) - k_refold_half_step;
            A_half_step(2, 2) = -k_off_talin_half_step(2) - k_unfold_half_step;
            A_half_step(1, 2) = k_unfold_half_step;
            A_half_step(2, 1) = k_refold_half_step;
            A_half_step(3, 3) = -k_off_non_talin_half_step;
            A_half_step = A_half_step / v;
            
            A_zero_step(1, 1) = -k_off_talin_zero_step(1) - k_refold_zero_step;
            A_zero_step(2, 2) = -k_off_talin_zero_step(2) - k_unfold_zero_step;
            A_zero_step(1, 2) = k_unfold_zero_step;
            A_zero_step(2, 1) = k_refold_zero_step;
            A_zero_step(3, 3) = -k_off_non_talin_zero_step;
            A_zero_step = A_zero_step / v;

            k_1 = A_zero_step * p_on_all(:, k);
            k_2 = A_half_step * ( p_on_all(:, k) + RK_half_step * k_1 );
            k_3 = A_half_step * ( p_on_all(:, k) + RK_half_step * k_2 );
            k_4 = A_step * ( p_on_all(:, k) + 2 * RK_half_step * k_3 );
            
            p_on_all(:, k + 1) = p_on_all(:, k) + RK_half_step / 3 * ( k_1 + 2 * k_2 + 2 * k_3 + k_4 );
           
            if(sum(p_on_all(:, k+1)) < 1e-06) 
                break
            end
        end
        
%         disp('Finished Runge-Kutta calculations')

        %%% Finding C constant %%%
                
        I = k_on * 2 * RK_half_step * sum( p_on_all(:, 1 : RK_N_steps), 'all' );
        C = k_on / ( v + I );
        p_on_all = C * p_on_all;
        
        %%% Calculating the average traction force %%%
        
        I_C = 2 * RK_half_step * sum( p_on_all(1, 1 : RK_N_steps), 'all' );
        denominator = 1 - number_added_sites_per_unfolded_clutch * I_C;
        if(denominator < 0.01)
            denominator = 0.01;
        end
        clutches_number_total = clutches_number / denominator;
        Clutches_on_zero_contour(:, j)          = clutches_number_total * 2 * RK_half_step * sum( p_on_all(:, 1 : RK_N_steps), 2 );
        F_average_zero_contour(:, j)            = clutches_number_total * 2 * RK_half_step * sum(    tabulated_clutch_forces_expanded(:, 1 : 2 : 2*RK_N_steps-1 )    .* p_on_all(:, 1 : RK_N_steps), 2 ) ./ Clutches_on_zero_contour(:, j);
        Loading_rate_average_zero_contour(:, j)                     = v * 2 * RK_half_step * sum( tabulated_force_derivative(:, 1 : 2 : 2*RK_N_steps-1 )    .* p_on_all(:, 1 : RK_N_steps), 2 ) ./ ( Clutches_on_zero_contour(:, j) / clutches_number_total );
        Loading_rate_average_square_zero_contour(:, j)            = v^2 * 2 * RK_half_step * sum( tabulated_force_derivative(:, 1 : 2 : 2*RK_N_steps-1 ).^2 .* p_on_all(:, 1 : RK_N_steps), 2 ) ./ ( Clutches_on_zero_contour(:, j) / clutches_number_total ).^2;
        Loading_rate_std_zero_contour(:, j) = sqrt(Loading_rate_average_square_zero_contour(:, j) - Loading_rate_average_zero_contour(:, j).^2);
    end
         
    semilogx(Young_modulus_zero_contour, F_average_zero_contour(1, :), 'r')
    hold on
    semilogx(Young_modulus_zero_contour, F_average_zero_contour(2, :), 'b')
    semilogx(Young_modulus_zero_contour, F_average_zero_contour(3, :), 'k')
    title('Tension of a single clutch')
    xlabel('Young modulus, Pa') 
    ylabel('Average force, pN') 
        
    figure(7)
    loglog(Young_modulus_zero_contour, Loading_rate_average_zero_contour(1, :), 'r')
    hold on
    errorbar(Young_modulus_zero_contour, Loading_rate_average_zero_contour(1, :), Loading_rate_std_zero_contour(1, :), 'r')
    errorbar(Young_modulus_zero_contour, Loading_rate_average_zero_contour(2, :), Loading_rate_std_zero_contour(2, :), 'b')
    errorbar(Young_modulus_zero_contour, Loading_rate_average_zero_contour(3, :), Loading_rate_std_zero_contour(3, :), 'k')
%     semilogx(Young_modulus_zero_contour, Loading_rate_average_zero_contour(1, :), 'r')
%     hold on
%     semilogx(Young_modulus_zero_contour, Loading_rate_average_zero_contour(2, :), 'b')
%     semilogx(Young_modulus_zero_contour, Loading_rate_average_zero_contour(3, :), 'k')
    title('Average loading rate of a single clutch')
    xlabel('Young modulus, Pa') 
    ylabel('Average loading rate, pN/s') 
    
    figure(8)
    Clutches_on_density_zero_contour = Clutches_on_zero_contour / (pi * a_2^2) * 1e+06;
    semilogx(Young_modulus_zero_contour, Clutches_on_zero_contour(1, :), 'r')
    hold on
    semilogx(Young_modulus_zero_contour, Clutches_on_zero_contour(2, :), 'b')
    semilogx(Young_modulus_zero_contour, Clutches_on_zero_contour(3, :), 'k')
    title('Average number of bonds')
    xlabel('Young modulus, Pa') 
    ylabel('Average number of bonds') 
        
%     figure(9)
%     semilogx(Young_modulus_zero_contour, clutches_number_catch_full_model * p_on_catch_full_model .* force_catch_full_model, 'g')
%     hold on
%     title('Total force')
%     xlabel('Young modulus, Pa') 
%     ylabel('Total force, pN') 
        
    figure(100)
    semilogx(Young_modulus_zero_contour, Cell_traction_zero_contour, 'g')
 
    figure(101)
    semilogx(Young_modulus_zero_contour, Actin_flow_zero_contour, 'g')
        
    if(break_flag)
        break
    end
end

